<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Consua Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Consua_Extention {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );

	}


	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}

		// Add Plugin actions
		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );

        // Register widget scripts
		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);

		// Register Widget Styles
		add_action( 'elementor/frontend/after_enqueue_styles', [ $this, 'enqueue_widget_styles' ] );
		add_action( 'elementor/editor/before_enqueue_scripts', [ $this, 'enqueue_widget_styles' ] );


        // category register
		add_action( 'elementor/elements/categories_registered',[ $this, 'Consua_elementor_widget_categories' ] );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'consua' ),
			'<strong>' . esc_html__( 'Consua Core', 'consua' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'consua' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'consua' ),
			'<strong>' . esc_html__( 'Consua Core', 'consua' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'consua' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'consua' ),
			'<strong>' . esc_html__( 'Consua Core', 'consua' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'consua' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
		 * Register Widget Styles
		 *
		 * Register custom styles required to run Saasland Core.
		 *
		 * @since 1.7.0
		 * @since 1.7.1 The method moved to this class.
		 *
		 * @access public
		 */
		
		public function enqueue_widget_styles() {
            wp_enqueue_style( 'consua-flaticons', CONSUA_PLUGDIRURI .'assets/fonts/flaticon-set.css');
		}



	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		require_once( CONSUA_ADDONS. '/widgets/consua-banner.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-process.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-choose-us.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-team.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-projects.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-request-callback.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-testimonial-carousel.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-feature.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-service.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-partner.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-faq.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-funfact.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-about.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-overview.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-pricing.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-video.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-contact-form.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-marketing.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-mission.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-service-menu.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-service-cta.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-service-brochure.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-blog.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-project-info.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-team-details.php' );
		require_once( CONSUA_ADDONS. '/widgets/consua-shape.php' );

		require_once( CONSUA_ADDONS. '/footer-widgets/consua-footer-about.php' );
		require_once( CONSUA_ADDONS. '/footer-widgets/consua-footer-navmenu.php' );
		require_once( CONSUA_ADDONS. '/footer-widgets/consua-newsletter.php' );
		require_once( CONSUA_ADDONS. '/footer-widgets/consua-footer-copyright.php' );

		require_once( CONSUA_ADDONS. '/widgets/consua-header.php' );
		

		// Register widget
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Process_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Banner_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Choose_Us_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Team_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Project_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Request_Callback_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Testomonial_Carousel_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Feature_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Service_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Partner_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Faq_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Funfactor_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_About_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Overview_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Pricing_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Footer_About_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Footer_Navmenu_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Footer_Newsletter_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Footer_Copyright_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Header_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Video_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Contact_Form_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Marketing_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Mission_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Service_Navmenu_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Service_Cta_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Service_Brochure_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Blog_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Project_Info_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Team_Deatils_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Elementor_Consua_Shape_Widget() );
	}

    public function widget_scripts() {
        wp_enqueue_script(
            'consua-frontend-script',
            CONSUA_PLUGDIRURI . 'assets/js/consua-frontend.js',
            array('jquery'),
            false,
            true
		);
	}
	

    function Consua_elementor_widget_categories( $elements_manager ) {
        $elements_manager->add_category(
            'consua_elements',
            [
                'title' => __( 'Consua', 'consua' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
        $elements_manager->add_category(
            'consua_footer_elements',
            [
                'title' => __( 'Consua Footer Elements', 'consua' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'consua_header_elements',
            [
                'title' => __( 'Consua Header Elements', 'consua' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

        $elements_manager->add_category(
            'consua_service_elements',
            [
                'title' => __( 'Consua Service Elements', 'consua' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

	}

}

Consua_Extention::instance();